//*********************************************************************
// Example5 is a bounded, constrained, MIP optimization problem. 
// 
// The goal of the optimization is to minimize the following expression subject to 
// the constraints listed below: 
//		5x1 + x2 + x3 + 2x4 + 3x5
//
//	Subject to these constraints:
//		x2 -  5x3 + x4 + 2x5 >= -2
//		5x1 - x2 + x5 >= 7
//		x1 + x2 + 6x3 + x4 >= 4
//
//		Varibles have the following bounds
//    0   <= x1 <= 100
//    0   <= x2 <= 100
//    0	<= x3 <= 100
//		0  <= x4 <= 100
//		0	<= x5 <= 100
//
// 
//   Decision variables x1, x2 and x3 are integer variables with step size of 1
//   Decision variables x4 and x5 are continuous variables
//
// The OptQuest.dll distributed with this example is a demo
// version with the following restrictions:
//		- Maximum variables = 7	
//		- Maximum constraints = 4
//		- Maximum requirements =  4
//		- Maximum iterations = 500
//

//*********************************************************************
#include "ocl.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

int main(int argc, char **argv)
{
	double ObjVal, DecVar[6];
	long   i, TotalIter, Example, status;

	if (argc != 2) 
	{
		TotalIter = 500;
	}
	else
		TotalIter = atoi(argv[1]);

	if (TotalIter > 500)
	{
		printf("Warning:  The Demo version is limited to 500 iterations\n");
		TotalIter = 500;
	}

	/* Allocating memory (setting up the problem) */

	Example = OCLSetup(5, 0, 3, 0, "MIN", 999999999);
	if (Example < 0) 
	{
		printf("OCLSetup error code %d\n", Example);
		exit(1);
	}

	/* Defining variables */

	OCLDefineVar(Example, 1,  0.0, OCLGetNull(Example), 100.0, "DIS", 1);
	OCLDefineVar(Example, 2,  0.0, OCLGetNull(Example), 100.0, "DIS", 1);
	OCLDefineVar(Example, 3,  0.0, OCLGetNull(Example), 100.0, "DIS", 1);
	OCLDefineVar(Example, 4,  0.0, OCLGetNull(Example), 100.0, "CON", 1);
	OCLDefineVar(Example, 5,  0.0, OCLGetNull(Example), 100.0, "CON", 1);
	
	/* Defining constraints */

   /* x2 -  5x3 + x4 + 2x5 >= -2 */
	OCLConsCoeff(Example, 1, 2, 1);
	OCLConsCoeff(Example, 1, 3,-5);
	OCLConsCoeff(Example, 1, 4, 1);
	OCLConsCoeff(Example, 1, 5, 2);
	OCLConsType (Example, 1, OCLGE);
	OCLConsRhs  (Example, 1, -2);

	/* 5x1 - x2 + x5 >= 7 */
	OCLConsCoeff(Example, 2, 1, 5);
	OCLConsCoeff(Example, 2, 2,-1);
	OCLConsCoeff(Example, 2, 5, 1);
	OCLConsType (Example, 2, OCLGE);
	OCLConsRhs  (Example, 2, 7);

	/* x1 + x2 + 6x3 + x4 >= 4 */
	OCLConsCoeff(Example, 3, 1, 1);
	OCLConsCoeff(Example, 3, 2, 1);
	OCLConsCoeff(Example, 3, 3, 6);
	OCLConsCoeff(Example, 3, 4, 1);
	OCLConsType (Example, 3, OCLGE);
	OCLConsRhs  (Example, 3, 4);

	// Define the objective function
	// 5x1 + x2 + x3 + 2x4 + 3x5
	OCLDefineObj(Example, 1, 5);
	OCLDefineObj(Example, 2, 1);
	OCLDefineObj(Example, 3, 1);
	OCLDefineObj(Example, 4, 2);
	OCLDefineObj(Example, 5, 3);

	// Call the LP solver
	status = OCLSolveLP(Example, &ObjVal, DecVar);
	if (status < 0) 
	{
		printf("OCLSolveLP error code %d\n", status);
		printf("\nPress any key to end ...\n");
		getchar();
		exit(1);
	}

	/* Display the  solution  */
	printf("The objective value is %6.2f\n", ObjVal);
	for (i=1; i<=5; i++) printf("x[%2d] = %6.2f\n",i, DecVar[i]);

	printf("\nPress any key to end ...\n");
	getchar();


	/* Free memory */	
	status = OCLGoodBye(Example);
	if (status < 0) 
	{
		printf("OCLGoodBye error code %d\n", status);
		exit(1);
	}
	return 0;
}
